<?php
//    ⭐️    ADIÇÃO    NECESSÁRIA:    Define    que    o    conteúdo    enviado    é    codificado    em    UTF-8
header('Content-Type:    text/html;    charset=utf-8');

//    Inicia    a    sessão
session_start();

//    Garanta    que    este    arquivo    existe    e    a    conexão    funciona.
include_once    'conexao.php';

//    ============================================================================
//    FUNÇÃO    NECESSÁRIA    PARA    bind_param    DINÂMICO
//    ============================================================================
function    refValues($arr)    {
    $refs    =    [];
    foreach    ($arr    as    $key    =>    $value)    {
    $refs[$key]    =    &$arr[$key];
    }
    return    $refs;
}


//    =================================================================================
//    1.    LÓGICA    DE    CONTROLE    DE    SESSÃO    E    INICIALIZAÇÃO    DE    VARIÁVEIS    DE    IDENTIFICAÇÃO
//    =================================================================================

//    Define    o    ID    do    usuário    e    o    perfil,    garantindo    um    valor    padrão    seguro
$usuario_id    =    (int)($_SESSION['usuario']['id']    ??    0);
$perfil    =    $_SESSION['usuario']['perfil']    ??    '';

//    Flags    de    controle
$is_professor    =    strtoupper(trim($perfil))    ===    'PROFESSOR';
$is_simulating    =    $_SESSION['simulando']    ??    false;

//    Determina    o    ID    a    ser    usado    na    consulta
//    Se    estiver    simulando    (e    o    ID    da    simulação    estiver    definido),    usa    o    ID    simulado,
//    caso    contrário,    usa    o    ID    do    usuário    logado.
$id_para_consulta    =    (int)($_SESSION['simulando_id']    ??    $usuario_id);

//    ---    VERIFICAÇÃO    DE    ACESSO    (Guard    Clauses)    ---
//    Se    não    for    professor    E    não    estiver    simulando,    redireciona.
if    (!$is_professor    &&    !$is_simulating)    {
    header('Location:    login.php');
    exit;
}
//    Se    o    ID    for    inválido    (mesmo    após    a    lógica    de    simulação),    redireciona.
if    ($id_para_consulta    <=    0)    {
    header('Location:    login.php');
    exit;
}

//    Nome    do    professor    alvo    (usando    $id_para_consulta,    pois    pode    ser    um    ID    simulado)
$nome_professor    =    '';
$sqlNome    =    "SELECT    nome    FROM    usuarios    WHERE    id    =    ?    LIMIT    1";

//    Verifica    se    a    conexão    $conn    existe    e    é    válida    antes    de    tentar    preparar
if    (isset($conn)    &&    $conn    instanceof    mysqli)    {
    $stmt    =    $conn->prepare($sqlNome);
    if    ($stmt)    {
    //    Usa    $id_para_consulta    no    bind
    $stmt->bind_param("i",    $id_para_consulta);
    $stmt->execute();
    $resNome    =    $stmt->get_result();

    if    ($resNome    &&    $resNome->num_rows    ===    1)    {
    $rowNome    =    $resNome->fetch_assoc();
    $nome_professor    =    $rowNome['nome'];
    }
    $stmt->close();
    }
}    else    {
    //    Loga    erro    se    a    conexão    não    estiver    definida/válida    (necessita    de    um    'require'    anterior)
    error_log("Erro:    Variável    \$conn    não    está    definida    ou    não    é    uma    instância    válida    de    mysqli.");
}

//    Ano    lectivo    e    Variáveis    Estáticas
$ano_lectivo_atual    =    '2025/2026';
$nome_republica    =    "REPÚBLICA    DE    ANGOLA";
$nome_governo    =    "GOVERNO    PROVINCIAL    DE    BENGUELA";
$nome_gabinete    =    "GABINETE    PROVINCIAL    DA    EDUCAÇÃO";
$nome_escola    =    "MAGISTÉRIO    Nº 0038     BLA    5 DE OUTUBRO - CATUMBELA";
$nome_sub_diretor    =    "";
$localidade    =    "CATUMBELA";

//    Variáveis    dinâmicas    da    turma    (inicializadas)
$classe_dinamica    =    '';
$nome_turma_dinamica    =    '';
$curso_area_dinamica    =    '';
$turno_dinamico    =    '';
$sala_dinamica    =    '';
$turma_completa    =    '';
$nome_disciplina    =    '';
$nome_turma_selecionada    =    '';
$prazo_fechado    =    false;
$alunos    =    [];
$notas_por_aluno    =    [];
$mac_por_aluno    =    [];

//    =================================================================================
//    2.    FUNÇÃO    BUSCAR    MAC
//    =================================================================================

/**
    *    Busca    o    MAC    consolidado    de    um    aluno    para    um    trimestre    específico.
    *
    *    @param    mysqli    $conn    A    conexão    com    o    banco    de    dados.
    *    @param    int    $aluno_id    O    ID    do    aluno.
    *    @param    int    $disciplina_id    O    ID    da    disciplina.
    *    @param    int    $turma_id    O    ID    da    turma.
    *    @param    int    $trimestre    O    trimestre    (1,    2,    ou    3).
    *    @return    float    O    valor    do    MAC    (0.0    se    não    encontrado).
    */
function    buscarMacConsolidado(mysqli    $conn,    int    $aluno_id,    int    $disciplina_id,    int    $turma_id,    int    $trimestre):    float    {
    //    Busca    o    MAC    do    aluno    para    o    trimestre    selecionado
    $sql    =    "SELECT    mac    FROM    avaliacoes_continuas
    WHERE    aluno_id    =    ?    AND    disciplina_id    =    ?    AND    turma_id    =    ?    AND    trimestre    =    ?";

    $stmt    =    $conn->prepare($sql);

    if    (!$stmt)    {
    error_log("Erro    de    preparação    SQL    (MAC):    "    .    $conn->error);
    return    0.0;
    }

    //    Bind:    4    inteiros    (IDs    e    trimestre)
    //    A    função    buscarMacConsolidado    recebe    os    tipos    já    tipados    (int)    no    PHP    7+
    $stmt->bind_param("iiii",    $aluno_id,    $disciplina_id,    $turma_id,    $trimestre);
    $stmt->execute();
    $result    =    $stmt->get_result();
    $stmt->close();

    if    ($result->num_rows    >    0)    {
    $row    =    $result->fetch_assoc();
    //    Garante    que    o    valor    retornado    é    float,    tratando    o    null    do    DB    como    0.0
    return    (float)($row['mac']    ??    0.0);
    }
    return    0.0;
}
//    =================================================================================
//    3.    INICIALIZAÇÃO    DE    VARIÁVEIS    DE    FILTRO    (LÊ    GET)    E    CARREGAMENTO    DE    LISTAS
//    =================================================================================
$turma_id    =    (int)($_GET['turma_id']    ??    0);
$disciplina_id    =    (int)($_GET['disciplina_id']    ??    0);
$trimestre_selecionado    =    (int)($_GET['trimestre']    ??    0);

//    Busca    de    Turmas    (Carrega    para    o    SELECT)
$turmas    =    [];
$sqlTurmas    =    "
    SELECT    DISTINCT    t.id,    t.nome,    t.classe,    t.sala,    t.curso,    t.turno
    FROM    turmas    t
    INNER    JOIN    atribuicoes    a    ON    t.id    =    a.turma_id
    WHERE    a.professor_id    =    ?
    ORDER    BY    t.classe,    t.nome
";
$stmt    =    $conn->prepare($sqlTurmas);
$stmt->bind_param("i",    $id_para_consulta);
$stmt->execute();
$turmas_res    =    $stmt->get_result();
$turmas    =    [];
while    ($row    =    $turmas_res->fetch_assoc())    {
    $partes    =    [];
    if    ($row['classe'])    $partes[]    =    $row['classe'];
    if    ($row['nome'])    $partes[]    =    $row['nome'];
    if    ($row['curso'])    $partes[]    =    $row['curso'];
    if    ($row['turno'])    $partes[]    =    $row['turno'];
    if    ($row['sala'])    $partes[]    =    $row['sala'];
    $row['descricao_formatada']    =    implode('-',    $partes);
    $turmas[]    =    $row;
}
$stmt->close();

//    Busca    de    Disciplinas    (Carrega    para    o    SELECT,    dependente    de    turma_id)
$disciplinas    =    [];
if    ($turma_id    >    0)    {
    $sqlDisciplinas    =    "
    SELECT    DISTINCT    d.id,    d.nome
    FROM    atribuicoes    a
    INNER    JOIN    disciplinas    d    ON    a.disciplina_id    =    d.id
    WHERE    a.professor_id    =    ?    AND    a.turma_id    =    ?
    ORDER    BY    d.nome
    ";
    $stmt    =    $conn->prepare($sqlDisciplinas);
    $stmt->bind_param("ii",    $id_para_consulta,    $turma_id);
    $stmt->execute();
    $disciplinas_res    =    $stmt->get_result();
    while    ($disciplina    =    $disciplinas_res->fetch_assoc())    {
    $disciplinas[]    =    $disciplina;
    }
    $stmt->close();
}

//    Mensagens    de    feedback    (Usa    SESSION    para    garantir    que    a    mensagem    não    suma    após    o    redirecionamento)
$mensagem_sucesso    =    $_SESSION['mensagem_sucesso']    ??    '';
$mensagem_erro    =    $_SESSION['mensagem_erro']    ??    '';
unset($_SESSION['mensagem_sucesso'],    $_SESSION['mensagem_erro']);


//    =================================================================================
//    4.    LÓGICA    DE    SALVAR    NOTAS    (INTEGRADA    E    ROBUSTA)
//    Garante    o    salvamento    e    o    redirecionamento    para    reexibir    os    dados    atualizados.
//    =================================================================================
  // ... Continuação da LÓGICA DE SALVAR NOTAS (bloco 4)

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_notas'])) { 
    
    // VARIÁVEIS DO POST SÃO NECESSÁRIAS AQUI PARA O CÓDIGO FUNCIONAR
    $disciplina_id_post = (int)($_POST['disciplina_id'] ?? 0);
    $turma_id_post = (int)($_POST['turma_id'] ?? 0);
    $trimestre_post = (int)($_POST['trimestre'] ?? 0);
    
    // ⭐️ AJUSTE: Remove a verificação de prazos. Agora, a permissão é baseada
    // APENAS no perfil e na validade das IDs essenciais do formulário.
    $pode_salvar = $is_simulating || 
                   ($is_professor && 
                    $turma_id_post > 0 && 
                    $disciplina_id_post > 0 && 
                    $trimestre_post > 0); 

    if (!$pode_salvar) {
        // Mensagem de erro alterada para refletir que a falha é por permissão ou dados incompletos
        $_SESSION['mensagem_erro'] = "Acesso negado para salvar notas. Verifique se você é um professor autorizado e se selecionou a Turma, Disciplina e Trimestre.";
        header("Location: mini_pauta_professor1.php?turma_id=$turma_id_post&disciplina_id=$disciplina_id_post&trimestre=$trimestre_post");
        exit;
    

    // Receber dados do formulário...
// ... O restante do seu código (transação, loop, etc.) continua aqui.
    }

    //    Receber    dados    do    formulário    (Usando    POST    para    consistência)
    $aluno_ids         =    $_POST['aluno_id']    ??    [];
    $disciplina_id_post    =    (int)($_POST['disciplina_id']    ??    0);
    $turma_id_post         =    (int)($_POST['turma_id']    ??    0);
    $trimestre_post     =    (int)($_POST['trimestre']    ??    0);
    $ano_lectivo    =    $ano_lectivo_atual;

    if    (empty($aluno_ids)    ||    $disciplina_id_post    <=    0    ||    $turma_id_post    <=    0    ||    $trimestre_post    <=    0)    {
    $_SESSION['mensagem_erro']    =    "Erro    ao    salvar:    dados    incompletos.";
    //    Redireciona    com    os    filtros    para    reexibir    a    pauta
    header("Location:    mini_pauta_professor1.php?turma_id=$turma_id_post&disciplina_id=$disciplina_id_post&trimestre=$trimestre_post");
    exit;
    }

    //    Iniciar    Transação
    $conn->begin_transaction();

    try    {
    
    //    Buscar    Classe    da    Turma    (Usando    $turma_id_post)
    $classe    =    "";
    $stmtC    =    $conn->prepare("SELECT    classe    FROM    turmas    WHERE    id=?");
    $stmtC->bind_param("i",    $turma_id_post);
    $stmtC->execute();
    $rC    =    $stmtC->get_result();
    if    ($rC->num_rows    >    0)    {
    $classe    =    $rC->fetch_assoc()['classe'];
    }
    $stmtC->close();

    //    Preparar    UPSERT    (REPLACE    INTO)
    $sql    =    "
    REPLACE    INTO    notas    (
    aluno_id,    turma_id,    disciplina_id,    ano_lectivo,    trimestre,    classe,
    mac,    npp,    npt,    mt,
    mt1,    mt2,    mt3,
    mfd,    mf,    neo,    nee,    mec,    exame,    nota_final,    observacao
    )    VALUES    (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
    ";

    $stmt    =    $conn->prepare($sql);
    if    (!$stmt)    {
    throw    new    Exception("Erro    preparar    REPLACE    INTO:    "    .    $conn->error);
    }

    //    Loop    dos    Alunos
    foreach    ($aluno_ids    as    $aluno_id)    {

    $aluno_id    =    (int)$aluno_id;
    //    ⭐️    REVISÃO:    Captura    o    array    de    notas    do    aluno    (ajuste    para    o    novo    name    do    input)
    $aluno_notas    =    $_POST['notas'][$aluno_id]    ??    [];    

    //    Coleta    notas    do    formulário
    //    MAC:    Usado    apenas    se    for    editável    (gestor),    senão    será    buscado    do    avaliacoes_continuas
    $mac_input    =    str_replace(",",    ".",    $_POST["mac_{$aluno_id}"]    ??    '0');
    //    ⭐️    CORREÇÃO:    Lê    NPP,    NPT,    NEO,    NEE    e    Observação    do    array    'notas'
    $npp_input    =    str_replace(",",    ".",    $aluno_notas["npp"]    ??    '0');
    $npt_input    =    str_replace(",",    ".",    $aluno_notas["npt"]    ??    '0');
    
    $neo_input    =    str_replace(",",    ".",    $aluno_notas["neo"]    ??    '0');    
    $nee_input    =    str_replace(",",    ".",    $aluno_notas["nee"]    ??    '0');    
    
    $obs_input    =    $aluno_notas["observacao"]    ??    '';

    //    Convertendo    para    float
    $mac_post    =    (float)$mac_input;    //    MAC    postado    (se    editável)
    $npp    =    (float)$npp_input; 
    $npt    =    (float)$npt_input;
    $neo_input_float    =    (float)$neo_input;
    $nee_input_float    =    (float)$nee_input;

    //    Variáveis    de    notas    (carregadas    ou    inicializadas)
    $mac    =    0.0;    $mt    =    0.0;
    $mt1    =    0.0;    $mt2    =    0.0;    $mt3    =    0.0;
    $mfd    =    $mf    =    $neo    =    $nee    =    $mec    =    $exame    =    $nota_final    =    0.0;

    //    ---    BUSCAR    DADOS    EXISTENTES    (Para    consolidar    MTs    anteriores    e    Notas    Finais)    ---
    //    Buscar    TODOS    os    trimestres    para    garantir    que    mt1,    mt2    e    mt3    estejam    corretos.
    $stmtB    =    $conn->prepare("SELECT    *    FROM    notas    WHERE    aluno_id=?    AND    disciplina_id=?    AND    turma_id=?    AND    ano_lectivo=?");
    $stmtB->bind_param("iiis",    $aluno_id,    $disciplina_id_post,    $turma_id_post,    $ano_lectivo);
    $stmtB->execute();
    $resB    =    $stmtB->get_result();

    $dados_existentes    =    [];
    while    ($row    =    $resB->fetch_assoc())    {
    $dados_existentes[(int)$row['trimestre']]    =    $row;
    }
    $stmtB->close();

    //    Recupera    MTs    e    Notas    Finais    existentes    de    TODOS    os    trimestres
    $mt1    =    (float)($dados_existentes[1]['mt']    ??    0.0);
    $mt2    =    (float)($dados_existentes[2]['mt']    ??    0.0);
    $mt3    =    (float)($dados_existentes[3]['mt']    ??    0.0);
    
    //    Recuperar    finais    existentes    (só    existem    no    registro    do    3º    trimestre)
    if    (isset($dados_existentes[3]))    {
    $mfd    =    (float)($dados_existentes[3]['mfd']    ??    0.0);
    $mf    =     (float)($dados_existentes[3]['mf']    ??    0.0);
    $neo    =    (float)($dados_existentes[3]['neo']    ??    0.0);
    $nee    =    (float)($dados_existentes[3]['nee']    ??    0.0);
    $mec    =    (float)($dados_existentes[3]['mec']    ??    0.0);
    $exame    =    (float)($dados_existentes[3]['exame']    ??    0.0);
    $nota_final    =    (float)($dados_existentes[3]['nota_final']    ??    0.0);
    }

    //    ====================================================================
    //    MAC:    Prioridade    1:    Valor    postado    (se    gestor);    Prioridade    2:    avaliacoes_continuas
    //    ====================================================================
    if    ($is_simulating)    {
    //    Se    estiver    simulando,    usa    o    valor    postado    do    MAC
    $mac    =    $mac_post;
    }    else    {
    //    Senão,    usa    o    MAC    consolidado    da    tabela    de    avaliação    contínua
    $mac    =    buscarMacConsolidado($conn,    $aluno_id,    $disciplina_id_post,    $turma_id_post,    $trimestre_post);
    }

    //    ====================================================================
    //    ⭐️    REFORÇO:    CÁLCULO    DA    MÉDIA    TRIMESTRAL    (MT)    -    MT    =    (MAC    +    NPP    +    NPT)    /    3
    //    Garantir    que    MAC,    NPP    e    NPT    estejam    sempre    presentes    no    cálculo
    //    ====================================================================
    $mt    =    ($mac    +    $npp    +    $npt)    /    3.0;
    $mt    =    round($mt,    2); 

    //    Atualiza    o    MT    do    trimestre    atual    com    o    valor    CALCULADO
    if    ($trimestre_post    ==    1)    $mt1    =    $mt;
    if    ($trimestre_post    ==    2)    $mt2    =    $mt;
    if    ($trimestre_post    ==    3)    $mt3    =    $mt;

    //    Atualizar    NEO    e    NEE    (somente    no    3º    trimestre)    com    os    valores    postados
    if    ($trimestre_post    ==    3)    {
    $neo    =    $neo_input_float;
    $nee    =    $nee_input_float;
    }

    $mt_principal    =    $mt;
    $npp_db_value    =    $npp;
    $npt_db_value    =    $npt;    //    Usando    $npt    como    float

    //    ==================    RECALCULAR    MÉDIAS    NO    3º    TRIMESTRE    ==================
    if    ($trimestre_post    ==    3)    {
    //    1.    Calcular    MFD
    $mts_validos    =    [];
    if    ($mt1    >    0)    $mts_validos[]    =    $mt1;
    if    ($mt2    >    0)    $mts_validos[]    =    $mt2;
    if    ($mt3    >    0)    $mts_validos[]    =    $mt3;

    if    (count($mts_validos)    ===    3)    {
    $mfd    =    array_sum($mts_validos)    /    3;
    }    else    {
    $mfd    =    0;
    }
    $mfd    =    round($mfd,    2);

    //    2.    Calcular    MEC
    $mec    =    ($neo    >    0    &&    $nee    >    0)    ?    (($neo    +    $nee)    /    2)    :    0;
    $mec    =    round($mec,    2);

    //    3.    Calcular    MF
    if    ($mfd    >    0)    {
    $mf    =    $mfd;    //    MF    default    é    MFD
    
    if    ($mfd    <    10    &&    ($mec    >    0    ||    $neo    >    0    ||    $nee    >    0))    {    //    Só    faz    exame/recurso    se    reprovado    ou    abaixo    de    10
    if    ($mec    >    0)    {
    $mf    =    ($mfd    *    0.6)    +    ($mec    *    0.4);
    }    else    if    ($neo    >    0    ||    $nee    >    0)    {
    //    Se    só    tem    uma    nota    de    recurso    (exame    normal    ou    especial)
    $nota_exame    =    $neo    >    0    ?    $neo    :    $nee;
     $mf    =    ($mfd    *    0.6)    +    ($nota_exame    *    0.4);
    }
    }
    }    else    {
    $mf    =    0;
    }
    $mf    =    round($mf,    2);

    //    4.    Nota    final    (arredondamento)
    if    ($mf    >=    9.5)    {    //    Arredonda    9.5    ou    mais    para    o    inteiro    mais    próximo    (10    ou    mais)
    $nota_final    =    ceil($mf);
    }    else    {
    $nota_final    =    $mf;
    }
    }
    //    ===================================================================================

    //    --------------------------------------------------------------------
    //    EXECUÇÃO    DO    REPLACE    INTO
    //    --------------------------------------------------------------------
    $stmt->bind_param(
    "iisissdddddddddddddds",
    $aluno_id,    $turma_id_post,    $disciplina_id_post,    $ano_lectivo,    $trimestre_post,    $classe,
    $mac,    $npp_db_value,    $npt_db_value,    $mt_principal,    //    MAC,    NPP,    NPT,    MT    (principal)
    $mt1,    $mt2,    $mt3,    //    MT1,    MT2,    MT3    (consolidados)
    $mfd,    $mf,    $neo,    $nee,    $mec,    $exame,    $nota_final,    //    Finais
    $obs_input    //    Coluna    de    Observação    (string)
    );

    if    (!$stmt->execute())    {
    throw    new    Exception("Erro    ao    executar    REPLACE    INTO    para    Aluno    ID    {$aluno_id}:    "    .    $stmt->error);
    }

    }    //    Fim    do    loop    foreach    ($aluno_ids    as    $aluno_id)


    //    5.    FINALIZAÇÃO    DA    TRANSAÇÃO
    $conn->commit();
    $_SESSION['mensagem_sucesso']    =    "Notas    salvas    e    consolidadas    com    sucesso!    Os    campos    MAC,    NPP,    NPT,    MT    e    Notas    Finais    foram    atualizados.";
    //    Redireciona    com    os    filtros    para    reexibir    a    pauta
    header("Location:    mini_pauta_professor1.php?turma_id=$turma_id_post&disciplina_id=$disciplina_id_post&trimestre=$trimestre_post");
    exit;

    }    catch    (Exception    $e)    {
    $conn->rollback();
    $_SESSION['mensagem_erro']    =    "Erro    ao    salvar    notas:    "    .    $e->getMessage();
    //    Redireciona    com    os    filtros    para    reexibir    a    pauta
    header("Location:    mini_pauta_professor1.php?turma_id=$turma_id_post&disciplina_id=$disciplina_id_post&trimestre=$trimestre_post");
    exit;
    }
}
//    Fim    da    Lógica    de    Salvar    Notas
//    =================================================================================


//    =================================================================================
//    6.    LÓGICA    DE    BLOQUEIO    DE    EDIÇÃO    (PRAZO)
//    Isso    define    as    variáveis    $mac_readonly,    $npp_readonly    e    $npt_readonly
//    =================================================================================
$data_hoje    =    date("Y-m-d");

//    Variáveis    de    Estado    (Inicia    como    liberado)
$npt_bloqueado    =    false;
$npp_bloqueado    =    false;
$motivo_bloqueio_npt    =    "";
$motivo_bloqueio_npp    =    "";

if    ($trimestre_selecionado    >    0)    {    //    Só    verifica    se    um    trimestre    foi    selecionado

    //    1.    VERIFICAR    PRAZO    DO    NPT
    //    -------------------------
    $tipo_nota    =    'NPT';
    $stmt_npt    =    $conn->prepare("
    SELECT    data_inicio,    data_fim
    FROM    prazos_lancamento
    WHERE    ano_lectivo    =    ?    AND    trimestre    =    ?    AND    tipo_nota    =    ?
    LIMIT    1
    ");
    $stmt_npt->bind_param("sis",    $ano_lectivo_atual,    $trimestre_selecionado,    $tipo_nota);
    $stmt_npt->execute();
    $resultado_npt    =    $stmt_npt->get_result()->fetch_assoc();

    if($resultado_npt)    {
    $inicio    =    $resultado_npt['data_inicio'];
    $fim    =    $resultado_npt['data_fim'];

    if    ($data_hoje    <    $inicio)    {
    $npt_bloqueado    =    true;
    $motivo_bloqueio_npt    =    "O    prazo    para    NPT    só    inicia    em    "    .    date("d/m/Y",    strtotime($inicio));
    }    elseif    ($data_hoje    >    $fim)    {
    $npt_bloqueado    =    true;
    $motivo_bloqueio_npt    =    "O    prazo    para    NPT    terminou    no    dia    "    .    date("d/m/Y",    strtotime($fim));
    }
    }    else    {
    $npt_bloqueado    =    true;
    $motivo_bloqueio_npt    =    "O    prazo    para    NPT    ainda    não    foi    definido.";
    }
    $stmt_npt->close();


    //    2.    VERIFICAR    PRAZO    DO    NPP
    //    -------------------------
    $tipo_nota    =    'NPP';
    $stmt_npp    =    $conn->prepare("
    SELECT    data_inicio,    data_fim
    FROM    prazos_lancamento
    WHERE    ano_lectivo    =    ?    AND    trimestre    =    ?    AND    tipo_nota    =    ?
    LIMIT    1
    ");
    $stmt_npp->bind_param("sis",    $ano_lectivo_atual,    $trimestre_selecionado,    $tipo_nota);
    $stmt_npp->execute();
    $resultado_npp    =    $stmt_npp->get_result()->fetch_assoc();

    if    ($resultado_npp)    {
    $inicio    =    $resultado_npp['data_inicio'];
    $fim    =    $resultado_npp['data_fim'];

    if    ($data_hoje    <    $inicio)    {
    $npp_bloqueado    =    true;
    $motivo_bloqueio_npp    =    "O    prazo    para    NPP    só    inicia    em    "    .    date("d/m/Y",    strtotime($inicio));
    }    elseif    ($data_hoje    >    $fim)    {
    $npp_bloqueado    =    true;
    $motivo_bloqueio_npp    =    "O    prazo    para    NPP    terminou    no    dia    "    .    date("d/m/Y",    strtotime($fim));
    }
    }    else    {
    $npp_bloqueado    =    true;
    $motivo_bloqueio_npp    =    "O    prazo    para    NPP    ainda    não    foi    definido.";
    }
    $stmt_npp->close();

}    else    {
    //    Se    nenhum    trimestre    foi    selecionado,    bloqueia    por    padrão
    $npt_bloqueado    =    true;
    $npp_bloqueado    =    true;
}


//    ---    DEFINIÇÃO    DAS    VARIÁVEIS    DE    CONTROLE    DE    EDIÇÃO    ---
//    MAC    só    é    editável    em    simulação
$mac_editavel    =    $is_simulating;
//    NPP/NPT    são    editáveis    se    o    prazo    estiver    aberto    OU    se    estiver    em    simulação
$npp_editavel    =    !$npp_bloqueado    ||    $is_simulating;
$npt_editavel    =    !$npt_bloqueado    ||    $is_simulating;

//    Define    o    atributo    readonly    e    estilo    para    o    MAC
$mac_readonly    =    $mac_editavel    ?    ''    :    'readonly';
$mac_style    =    $mac_editavel    ?    'background:#fff;'    :    'background:#eee;';

//    Define    o    atributo    readonly    e    estilo    para    NPP    e    NPT
$npp_readonly    =    $npp_editavel    ?    ''    :    'readonly';
$npp_style    =    $npp_editavel    ?    'background:#fff;'    :    'background:#eee;';

$npt_readonly    =    $npt_editavel    ?    ''    :    'readonly';
$npt_style    =    $npt_editavel    ?    'background:#fff;'    :    'background:#eee;';
//    =================================================================================


//    =================================================================================
//    7.    DETALHES    DA    TURMA    E    CARREGAMENTO    DE    ALUNOS    E    NOTAS    PARA    EXIBIÇÃO
//    =================================================================================

//    Detalhes    da    Turma    para    o    Cabeçalho    (Usa    $turma_id    vindo    do    GET)
if    ($turma_id    >    0)    {
    foreach    ($turmas    as    $turma)    {
    if    ($turma['id']    ==    $turma_id)    {
    $nome_turma_selecionada    =    $turma['descricao_formatada'];
    
    $partes_turma    =    explode('-',    $nome_turma_selecionada);
    $classe_dinamica    =    $partes_turma[0]    ??    '';
    $nome_turma_dinamica    =    $partes_turma[1]    ??    '';
    $curso_area_dinamica    =    $partes_turma[2]    ??    '';
    $turno_dinamico    =    $partes_turma[3]    ??    '';
    $sala_dinamica    =    $partes_turma[4]    ??    '';

    $turma_completa    =    "Turma:    "    .    $nome_turma_dinamica    .    "    -    "    .    $turno_dinamico    .    "    sala    nº    "    .    $sala_dinamica;
    
    break;
    }
    }
}

//    Busca    de    nome    da    Disciplina
if    ($disciplina_id    >    0)    {
    $sqlNomeDisciplina    =    "SELECT    nome    FROM    disciplinas    WHERE    id    =    ?";
    $stmt    =    $conn->prepare($sqlNomeDisciplina);
    $stmt->bind_param("i",    $disciplina_id);
    $stmt->execute();
    $res    =    $stmt->get_result();
    if    ($res->num_rows    >    0)    {
    $row    =    $res->fetch_assoc();
    $nome_disciplina    =    $row['nome'];
    }
    $stmt->close();
}

//    Carrega    Alunos    e    Notas    para    a    Mini-Pauta
if    ($turma_id    >    0    &&    $disciplina_id    >    0    &&    $trimestre_selecionado    >    0)    {
    //    Verifica    se    o    trimestre    já    foi    fechado
    $sqlCheckFecho    =    "SELECT    COUNT(*)
    FROM    fecho_trimestre
    WHERE    turma_id    =    ?    AND    disciplina_id    =    ?    AND    trimestre    =    ?";
    $stmtCheckFecho    =    $conn->prepare($sqlCheckFecho);
    $stmtCheckFecho->bind_param("iii",    $turma_id,    $disciplina_id,    $trimestre_selecionado);
    $stmtCheckFecho->execute();
    $count    =    $stmtCheckFecho->get_result()->fetch_row()[0];
    $stmtCheckFecho->close();
    $prazo_fechado    =    ($count    >    0);

    //    Buscar    alunos    da    turma
    $sqlAlunos    =    "SELECT    id,    nome    FROM    alunos    WHERE    turma_id    =    ?    ORDER    BY    nome";
    $stmtAlunos    =    $conn->prepare($sqlAlunos);
    $stmtAlunos->bind_param("i",    $turma_id);
    $stmtAlunos->execute();
    $alunos_res    =    $stmtAlunos->get_result();
    $alunos_data    =    $alunos_res->fetch_all(MYSQLI_ASSOC);
    $stmtAlunos->close();

    $alunos    =    $alunos_data;

    //    Buscar    notas    já    lançadas
    $aluno_ids_data    =    array_column($alunos_data,    'id');
    if    (!empty($aluno_ids_data))    {
    
    $in_clause    =    str_repeat('?,',    count($aluno_ids_data)    -    1)    .    '?';

    //    -------------------------------------------------------------
    //    1.    BUSCA    MAC    DA    TABELA    AVALIACOES_CONTINUAS    (Recente)
    //    -------------------------------------------------------------
    $sqlMAC    =    "
    SELECT    aluno_id,    trimestre,    mac
    FROM    avaliacoes_continuas
    WHERE    aluno_id    IN    ($in_clause)
    AND    disciplina_id    =    ?
    AND    turma_id    =    ?
    ";
    $params_types_mac    =    str_repeat('i',    count($aluno_ids_data))    .    'ii';
    $params_mac    =    array_merge($aluno_ids_data,    [$disciplina_id,    $turma_id]);

    $stmtMAC    =    $conn->prepare($sqlMAC);
    call_user_func_array([$stmtMAC,    'bind_param'],    refValues(array_merge([$params_types_mac],    $params_mac)));
    $stmtMAC->execute();
    $resultMAC    =    $stmtMAC->get_result();
    
    while    ($mac_row    =    $resultMAC->fetch_assoc())    {
    $mac_por_aluno[$mac_row['aluno_id']][$mac_row['trimestre']]    =    (float)$mac_row['mac'];
    }
    $stmtMAC->close();
    
    //    -------------------------------------------------------------
    //    2.    BUSCAR    OUTRAS    NOTAS    (NPP,    NPT,    MT,    etc.)    DA    TABELA    NOTAS
    //    -------------------------------------------------------------
    $sqlNotas    =    "
    SELECT    *    FROM    notas
    WHERE    aluno_id    IN    ($in_clause)
    AND    disciplina_id    =    ?
    AND    turma_id    =    ?
    ";
    $params_types    =    str_repeat('i',    count($aluno_ids_data))    .    'ii';
    $params    =    array_merge($aluno_ids_data,    [$disciplina_id,    $turma_id]);

    $stmtNotas    =    $conn->prepare($sqlNotas);
    call_user_func_array([$stmtNotas,    'bind_param'],    refValues(array_merge([$params_types],    $params)));
    $stmtNotas->execute();
    $resultNotas    =    $stmtNotas->get_result();
    
    while    ($nota    =    $resultNotas->fetch_assoc())    {
    $aluno    =    $nota['aluno_id'];
    $trimestre    =    $nota['trimestre'];
    
    //    Unifica    as    notas    da    tabela    'notas'
    $notas_por_aluno[$aluno][$trimestre]    =    $nota;
    
    //    ⭐️    REVISÃO:    Garante    que    NPP    e    NPT    sejam    pelo    menos    0.0    se    for    NULL    no    DB
    $notas_por_aluno[$aluno][$trimestre]['npp']    =    (float)($nota['npp']    ??    0.0);
    $notas_por_aluno[$aluno][$trimestre]['npt']    =    (float)($nota['npt']    ??    0.0);
    
    //    SOBREPÕE    o    MAC    (prioriza    a    tabela    avaliacoes_continuas)
    if    (isset($mac_por_aluno[$aluno][$trimestre]))    {
    $notas_por_aluno[$aluno][$trimestre]['mac']    =    $mac_por_aluno[$aluno][$trimestre];
    }    else    {
     //    Se    não    encontrou    MAC    em    avaliacoes_continuas,    garante    que    o    MAC    da    tabela    notas    seja    usado    ou    0.0
     $notas_por_aluno[$aluno][$trimestre]['mac']    =    (float)($nota['mac']    ??    0.0);
    }
    
    //    -------------------------------------------------------------
    //    ⭐️    LÓGICA    REFORÇADA:    RECALCULAR    MT    PARA    EXIBIÇÃO    (MT    =    (MAC    +    NPP    +    NPT)    /    3)
    //    Isso    garante    que    o    MT    exibido    esteja    sempre    correto    após    o    salvamento.
    //    -------------------------------------------------------------
    $mac_lido_exibicao    =    (float)($notas_por_aluno[$aluno][$trimestre]['mac']    ??    0.0);
    $npp_lido_exibicao    =    (float)($notas_por_aluno[$aluno][$trimestre]['npp']    ??    0.0);
    $npt_lido_exibicao    =    (float)($notas_por_aluno[$aluno][$trimestre]['npt']    ??    0.0);
    
    //    Recálculo    usando    os    valores    lidos    (incluindo    MAC    da    avaliacao_continua)
    if    ($mac_lido_exibicao    >    0.0    ||    $npp_lido_exibicao    >    0.0    ||    $npt_lido_exibicao    >    0.0)    { 
    $mt_recalculado    =    ($mac_lido_exibicao    +    $npp_lido_exibicao    +    $npt_lido_exibicao)    /    3.0;
    $notas_por_aluno[$aluno][$trimestre]['mt']    =    round($mt_recalculado,    2);
    }    else    {
    $notas_por_aluno[$aluno][$trimestre]['mt']    =    0.0;
    }
    //    -------------------------------------------------------------
    }
    $stmtNotas->close();
    }
}

//    Variáveis    injetadas    no    JavaScript
$notas_js    =    json_encode($notas_por_aluno);
$trimestre_js    =    (int)($trimestre_selecionado);
?>
<!DOCTYPE    html>
<html    lang="pt-br">
<head>
    <meta    charset="UTF-8">
    <meta    name="viewport"    content="width=device-width,    initial-scale=1.0">
    <title>Mini    Pauta    -    Professor</title>
    <link rel="stylesheet" href="css/tailwind.min.css">
<link rel="stylesheet" href="css/estilo.css">

<style>
/* ==========================================================
   SISMAG – DESIGN OFICIAL RESPONSIVO (ESCOLAR MODERNO)
   ========================================================== */

/* ===== BASE ===== */
body {
    font-family: 'Segoe UI', Tahoma, Arial, sans-serif;
    background: linear-gradient(180deg, #f1f5f9, #e2e8f0);
    color: #1f2937;
}

.container-pauta {
    max-width: 100%;
    margin: 1rem;
    background: #ffffff;
    padding: 1rem;
    border-radius: 14px;
    box-shadow: 0 6px 16px rgba(0,0,0,0.08);
}

/* ===== CABEÇALHO ESCOLA ===== */
.cabecalho-escola h1,
.cabecalho-escola h2,
.cabecalho-escola h3,
.cabecalho-escola p {
    line-height: 1.3;
}

/* ===== FORMULÁRIOS ===== */
select, input[type="text"] {
    transition: all 0.2s ease;
}

select:focus,
input[type="text"]:focus {
    outline: none;
    border-color: #2563eb;
    box-shadow: 0 0 0 2px rgba(37,99,235,0.2);
}

/* ===== TABELA DESKTOP ===== */
.tabela-pauta {
    width: 100%;
    border-collapse: collapse;
}

.tabela-pauta th,
.tabela-pauta td {
    border: 1px solid #e5e7eb;
    padding: 8px;
    font-size: 12px;
}

.tabela-pauta th {
    background: #1e3a8a;
    color: #ffffff;
    text-align: center;
    text-transform: uppercase;
    font-weight: 700;
    position: sticky;
    top: 0;
    z-index: 10;
}

.tabela-pauta tbody tr:hover {
    background: #eef2ff;
}

/* ===== COLUNAS ===== */
.coluna-nome-aluno-tela {
    min-width: 220px;
    font-weight: 600;
}

/* ===== INPUTS DE NOTAS ===== */
.nota-input {
    width: 48px;
    height: 32px;
    font-size: 13px;
    text-align: center;
    border: 1px solid #cbd5f5;
    border-radius: 8px;
    background: #f8fafc;
}

/* ===== NOTAS CALCULADAS ===== */
.mt-display,
.nota-final-calc {
    font-weight: 700;
    background: #e0f2fe !important;
    color: #1e40af;
}

/* ===== CORES DE RESULTADO ===== */
.nota-aprovada {
    color: #16a34a;
    font-weight: 700;
}

.nota-reprovada {
    color: #dc2626;
    font-weight: 700;
}

.estado-final-aprovado {
    background: #dcfce7;
    color: #166534;
    font-weight: 700;
    text-align: center;
}

.estado-final-reprovado {
    background: #fee2e2;
    color: #991b1b;
    font-weight: 700;
    text-align: center;
}

.estado-final-incompleto {
    background: #fef3c7;
    color: #92400e;
    font-weight: 700;
    text-align: center;
}

/* ===== BOTÕES ===== */
button {
    border-radius: 10px;
    transition: all 0.2s ease;
}

button:hover {
    transform: translateY(-1px);
}

/* ==========================================================
   📱 RESPONSIVO MOBILE – PADRÃO SISMAG
   ========================================================== */
@media (max-width: 768px) {

    .container-pauta {
        margin: 0.5rem;
        padding: 0.75rem;
    }

    /* Esconde cabeçalho da tabela */
    .tabela-pauta thead {
        display: none;
    }

    .tabela-pauta,
    .tabela-pauta tbody,
    .tabela-pauta tr,
    .tabela-pauta td {
        display: block;
        width: 100%;
    }

    .tabela-pauta tr {
        background: #ffffff;
        margin-bottom: 1rem;
        padding: 0.75rem;
        border-radius: 14px;
        box-shadow: 0 4px 10px rgba(0,0,0,0.1);
    }

    .tabela-pauta td {
        border: none;
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 6px 0;
        font-size: 13px;
    }

    .tabela-pauta td::before {
        content: attr(data-label);
        font-weight: 700;
        color: #1e3a8a;
    }

    .nota-input {
        width: 72px;
    }
}

/* ==========================================================
   🖨️ IMPRESSÃO
   ========================================================== */
@media print {
    body {
        background: #ffffff;
    }

    .container-pauta {
        box-shadow: none;
        border-radius: 0;
        margin: 0;
        padding: 0;
    }

    .no-print {
        display: none;
    }

    .nota-input {
        display: none;
    }

    .print-value::after {
        content: attr(data-nota-valor);
        display: block;
    }
}
</style>

</head>
<body    class="bg-gray-100">

<div    class="container-pauta    print-area">
    <div    class="no-print">
    <?php    if    ($mensagem_sucesso):    ?>
    <div    class="bg-green-100    border    border-green-400    text-green-700    px-4    py-3    rounded    relative    mb-4"    role="alert">
    <span    class="block    sm:inline"><?=    htmlspecialchars($mensagem_sucesso)    ?></span>
    </div>
    <?php    endif;    ?>
    <?php    if    ($mensagem_erro):    ?>
    <div    class="bg-red-100    border    border-red-400    text-red-700    px-4    py-3    rounded    relative    mb-4"    role="alert">
    <span    class="block    sm:inline"><?=    htmlspecialchars($mensagem_erro)    ?></span>
    </div>
    <?php    endif;    ?>

    <form    method="GET"    action="mini_pauta_professor1.php"    class="bg-white    p-6    rounded-lg    shadow-md    mb-6    border    border-gray-200">
    <h2    class="text-xl    font-bold    mb-4">Filtros    da    Mini    Pauta</h2>
    <div    class="flex    flex-wrap    -mx-3    mb-4">
    <div    class="w-full    md:w-1/3    px-3    mb-6    md:mb-0">
    <label    class="block    uppercase    tracking-wide    text-gray-700    text-xs    font-bold    mb-2"    for="turma_id">
    Turma:
    </label>
    <select    name="turma_id"    id="turma_id"    class="appearance-none    block    w-full    bg-gray-200    text-gray-700    border    border-gray-200    rounded    py-3    px-4    leading-tight    focus:outline-none    focus:bg-white    focus:border-gray-500"    onchange="this.form.submit()">
    <option    value="0">Selecione    a    Turma</option>
    <?php    foreach    ($turmas    as    $turma):    ?>
    <option    value="<?=    $turma['id']    ?>"    <?=    ($turma_id    ==    $turma['id'])    ?    'selected'    :    ''    ?>>
    <?=    htmlspecialchars($turma['descricao_formatada'])    ?>
    </option>
    <?php    endforeach;    ?>
    </select>
    </div>

    <div    class="w-full    md:w-1/3    px-3    mb-6    md:mb-0">
    <label    class="block    uppercase    tracking-wide    text-gray-700    text-xs    font-bold    mb-2"    for="disciplina_id">
    Disciplina:
    </label>
    <select    name="disciplina_id"    id="disciplina_id"    class="appearance-none    block    w-full    bg-gray-200    text-gray-700    border    border-gray-200    rounded    py-3    px-4    leading-tight    focus:outline-none    focus:bg-white    focus:border-gray-500"    onchange="this.form.submit()">
    <option    value="0">Selecione    a    Disciplina</option>
    <?php    foreach    ($disciplinas    as    $disciplina):    ?>
    <option    value="<?=    $disciplina['id']    ?>"    <?=    ($disciplina_id    ==    $disciplina['id'])    ?    'selected'    :    ''    ?>>
    <?=    htmlspecialchars($disciplina['nome'])    ?>
    </option>
    <?php    endforeach;    ?>
    </select>
    </div>
    <div    class="w-full    md:w-1/3    px-3">
    <label    class="block    uppercase    tracking-wide    text-gray-700    text-xs    font-bold    mb-2"    for="trimestre">
    Trimestre:
    </label>
    <select    name="trimestre"    id="trimestre"    class="appearance-none    block    w-full    bg-gray-200    text-gray-700    border    border-gray-200    rounded    py-3    px-4    leading-tight    focus:outline-none    focus:bg-white    focus:border-gray-500"    onchange="this.form.submit()">
    <option    value="0">Selecione    o    Trimestre</option>
    <option    value="1"    <?=    ($trimestre_selecionado    ==    1)    ?    'selected'    :    ''    ?>>1º    Trimestre</option>
    <option    value="2"    <?=    ($trimestre_selecionado    ==    2)    ?    'selected'    :    ''    ?>>2º    Trimestre</option>
    <option    value="3"    <?=    ($trimestre_selecionado    ==    3)    ?    'selected'    :    ''    ?>>3º    Trimestre</option>
    </select>
    </div>
    </div>
    <?php    if    ($turma_id    >    0    &&    $disciplina_id    >    0    &&    $trimestre_selecionado    >    0):    ?>
    <div    class="mt-4    text-center">
    <?php    if    ($prazo_fechado):    ?>
    <p    class="text-red-600    font-bold">⚠️    O    lançamento    para    este    trimestre/disciplina    está    **FECHADO**.</p>
    <?php    elseif    ($is_simulating):    ?>
    <p    class="text-yellow-600    font-bold">🔄    Você    está    em    **MODO    DE    SIMULAÇÃO**    (Gestor).    A    edição    está    liberada.</p>
    <?php    else:    ?>
    <p    class="text-green-600    font-bold">🟢    Prazo    de    Lançamento    **ABERTO**.</p>
    <?php    endif;    ?>
    </div>
    <?php    endif;    ?>
    </form>
    </div>

    <?php    if    ($turma_id    >    0    &&    $disciplina_id    >    0    &&    $trimestre_selecionado    >    0):    ?>

    <div    class="cabecalho-escola    text-center    mb-6">
    <h1    class="text-xs    font-bold"><?=    htmlspecialchars($nome_republica)    ?></h1>
    <p    class="text-xs"><?=    htmlspecialchars($nome_governo)    ?></p>
    <p    class="text-xs"><?=    htmlspecialchars($nome_gabinete)    ?></p>
    <h2    class="text-sm    font-bold    mt-2"><?=    htmlspecialchars($nome_escola)    ?></h2>
    <h3    class="text-base    font-bold    mt-4">MINI    PAUTA    DE    AVALIAÇÃO    DO    <?=    $trimestre_selecionado    ?>º    TRIMESTRE</h3>
    </div>

    <div    class="cabecalho-pauta-resumo    text-left    text-sm    mb-4    border    p-2">
    <p><strong>Ano    Lectivo:</strong>    <?=    htmlspecialchars($ano_lectivo_atual)    ?></p>
    <p><strong>Classe:</strong>    <?=    htmlspecialchars($classe_dinamica)    ?></p>
    <p><strong>Curso/Área:</strong>    <?=    htmlspecialchars($curso_area_dinamica)    ?></p>
    <p><strong>Disciplina:</strong>    <?=    htmlspecialchars($nome_disciplina)    ?></p>
    <p><strong>Turma:</strong>    <?=    htmlspecialchars($nome_turma_dinamica)    ?>    |    <strong>Sala:</strong>    <?=    htmlspecialchars($sala_dinamica)    ?>    |    <strong>Turno:</strong>    <?=    htmlspecialchars($turno_dinamico)    ?></p>
    <p><strong>Professor(a):</strong>    <?=    htmlspecialchars($nome_professor)    ?></p>
    </div>

    <form    method="POST"    action="mini_pauta_professor1.php"    class="form-pauta">
    <input    type="hidden"    name="turma_id"    value="<?=    $turma_id    ?>">
    <input    type="hidden"    name="disciplina_id"    value="<?=    $disciplina_id    ?>">
    <input    type="hidden"    name="trimestre"    value="<?=    $trimestre_selecionado    ?>">
    <input    type="hidden"    name="salvar_notas"    value="1">

    <div    class="overflow-x-auto">
    <table    class="tabela-pauta    min-w-full">
    <thead>
    <tr>
    <th    rowspan="2"    class="w-8">Nº</th>
    <th    rowspan="2">NOME    COMPLETO    DO    ALUNO(A)</th>
    <th    colspan="4">NOTAS    DO    <?=    $trimestre_selecionado    ?>º    TRIMESTRE</th>
    <?php    if    ($trimestre_selecionado    ==    3):    ?>
    <th    colspan="2">Exames</th>
    <th    colspan="4">RESULTADOS    FINAIS</th>
    <?php    endif;    ?>
    <th    rowspan="2"    class="w-20">OBSERVAÇÃO</th>
    </tr>
    <tr>
    <th>MAC</th>
    <th>NPP</th>
    <th>NPT</th>
    <th>MT</th>
    <?php    if    ($trimestre_selecionado    ==    3):    ?>
    <th>MFD</th>
    <th>NEO</th>
    <th>NEE</th>
    <th>MEC</th>
    <th>MF</th>
    <th>ESTADO</th>
    <?php    endif;    ?>
    </tr>
    </thead>
    <tbody>
    <?php    
    $contador    =    1;
    foreach    ($alunos    as    $aluno):    
    $aluno_id    =    $aluno['id'];

    //    Coleta    das    notas
    $nota_atual    =    $notas_por_aluno[$aluno_id][$trimestre_selecionado]    ??    [];
    //    ⭐️    REVISÃO:    Garante    que    os    valores    lidos    sejam    pelo    menos    0.0    para    exibição
    $mac_salvo    =    (float)($nota_atual['mac']    ??    0.0);    
    $npp_salvo    =    (float)($nota_atual['npp']    ??    0.0);
    $npt_salvo    =    (float)($nota_atual['npt']    ??    0.0);
    $mt_salvo     =    (float)($nota_atual['mt']    ??    0.0);    //    Este    MT    já    vem    recalculado    da    busca
    $obs_salvo    =    $nota_atual['observacao']    ??    '';
    
    //    Colunas    NEO/NEE/MFD/MEC/MF/Estado    para    o    3º    Trimestre
    $neo_salvo    =    (float)($nota_atual['neo']    ??    0.0);
    $nee_salvo    =    (float)($nota_atual['nee']    ??    0.0);

    //    Classes    de    cor    -    Usando    o    valor    float    salvo
    $determina_classe    =    function($nota)    {
    if    ($nota    ===    null    ||    $nota    ===    0.0)    return    '';
    return    ($nota    >=    10.0)    ?    'nota-aprovada'    :    'nota-reprovada';
    };
    
    $mac_classe    =    $determina_classe($mac_salvo);
    $npp_classe    =    $determina_classe($npp_salvo);
    $npt_classe    =    $determina_classe($npt_salvo);
    $mt_classe     =    $determina_classe($mt_salvo);

    //    Cálculo    de    notas    finais    (3º    trimestre)    -    Mantido    inalterado,    usa    os    valores    já    calculados
    $mfd    =    $mec    =    $mf    =    null;
    $estado_final    =    '---';
    $mf_display    =    '---';
    $estado_classe    =    'estado-final-incompleto';
    
    if    ($trimestre_selecionado    ==    3)    {
    $mfd    =    (float)($nota_atual['mfd']    ??    0.0);
    $mf    =    (float)($nota_atual['mf']    ??    0.0);    //    MF    antes    do    arredondamento    (ou    MFD    se    não    houve    exame)
    $neo    =    $neo_salvo;
    $nee    =    $nee_salvo;
    $mec    =    (float)($nota_atual['mec']    ??    0.0);
    $nota_final    =    (float)($nota_atual['nota_final']    ??    0.0);    //    Nota    final    após    arredondamento

    //    Determinar    estado    e    display
    if    ($nota_final    >=    10.0)    {
    $estado_final='APROVADO';
    $estado_classe='estado-final-aprovado';
    $mf_display    =    number_format(round($nota_final),0,',','.');    //    Exibe    inteiro    arredondado
    }    elseif    ($nota_final    >    0)    {
    $estado_final='REPROVADO';
    $estado_classe='estado-final-reprovado';
    $mf_display    =    number_format($nota_final,2,',','.');    //    Exibe    com    2    casas    decimais
    }
    
    $mfd_classe    =    $determina_classe($mfd);
    $neo_classe    =    $determina_classe($neo_salvo);
    $nee_classe    =    $determina_classe($nee_salvo);
    $mec_classe    =    $determina_classe($mec);
    $mf_classe    =    $determina_classe($nota_final);    //    Usamos    nota_final    para    determinar    a    cor
    }
    ?>
    <tr    data-aluno-id="<?=    $aluno_id    ?>"    class="hover:bg-gray-100">
    <td><?=    $contador++    ?></td>
    <td    class="coluna-nome-aluno-tela">
    <?=    htmlspecialchars($aluno['nome_completo']    ??    $aluno['nome'])    ?>
    <input    type="hidden"    name="aluno_id[]"    value="<?=    $aluno_id    ?>">
    </td>

    <td>
    <input    type="text"    
    name="mac_<?=    $aluno_id    ?>"    
    value="<?=    $mac_salvo    >    0    ?    str_replace('.',    ',',    number_format($mac_salvo,2))    :    ''    ?>"    
    class="nota-input    text-center    <?=    $mac_classe    ?>    mac-input"    
    onkeypress="return    onlyNumbersAndComma(event)"
    <?=    $mac_readonly    ?>    style="<?=    $mac_style    ?>"
    data-aluno-id="<?=    $aluno_id    ?>"
    data-mac-valor="<?=    $mac_salvo    ?>"    >
    </td>

    <td>
    <input    type="text"    
    name="notas[<?=    $aluno_id    ?>][npp]"    
    value="<?=    $npp_salvo    >    0    ?    str_replace('.',    ',',    $npp_salvo)    :    ''    ?>"    
    class="nota-input    text-center    <?=    $npp_classe    ?>    nota-npp-input"    
    data-aluno-id="<?=    $aluno_id    ?>"
    data-trimestre="<?=    $trimestre_selecionado    ?>"
    onkeypress="return    onlyNumbersAndComma(event)"
    <?=    $npp_readonly    ?>    style="<?=    $npp_style    ?>"
    <?=    $npp_readonly    ?    'title="'.$motivo_bloqueio_npp.'"'    :    ''    ?>>
    </td>

    <td>
    <input    type="text"    
    name="notas[<?=    $aluno_id    ?>][npt]"    
    value="<?=    $npt_salvo    >    0    ?    str_replace('.',    ',',    $npt_salvo)    :    ''    ?>"    
    class="nota-input    text-center    <?=    $npt_classe    ?>    nota-npt-input"    
    data-aluno-id="<?=    $aluno_id    ?>"
    data-trimestre="<?=    $trimestre_selecionado    ?>"
    onkeypress="return    onlyNumbersAndComma(event)"
    <?=    $npt_readonly    ?>    style="<?=    $npt_style    ?>"
    <?=    $npt_readonly    ?    'title="'.$motivo_bloqueio_npt.'"'    :    ''    ?>>
    </td>

    <td>
    <span    data-nota-valor="<?=    $mt_salvo    >    0    ?    number_format($mt_salvo,2,',','.'):'---'    ?>"    class="print-value    nota-calculada    <?=    $mt_classe    ?>">
    <input    type="text"    
    name="notas[<?=    $aluno_id    ?>][mt_display]"
    id="mt_display_<?=    $aluno_id    ?>"
    value="<?=    $mt_salvo    >    0    ?    str_replace('.',    ',',    $mt_salvo)    :    ''    ?>"    
    class="nota-input    text-center    <?=    $mt_classe    ?>    mt-display"    
    readonly    style="background:#eee;">
    </span>
    </td>

    <?php    if    ($trimestre_selecionado==3):    ?>
    <td    id="mfd_<?=    $aluno_id    ?>"    
    data-nota-valor="<?=    $mfd    >    0    ?    number_format($mfd,2,',','.'):'---'    ?>"    
    class="print-value    nota-calculada    <?=    $mfd_classe    ?>">
    <?=    $mfd    >    0    ?    number_format($mfd,2,',','.'):'---'    ?>
    </td>
    
    <td>
    <input    type="text"    
    name="notas[<?=    $aluno_id    ?>][neo]"    
    id="neo_<?=    $aluno_id    ?>"    
    value="<?=    $neo_salvo    >    0    ?    str_replace('.',    ',',    $neo_salvo)    :    ''    ?>"    
    class="nota-input    text-center    <?=    $neo_classe    ?>    neo-input"    
    maxlength="5"    placeholder="0,00"    
    data-aluno-id="<?=    $aluno_id    ?>"
    onkeypress="return    onlyNumbersAndComma(event)">
    </td>
    
    <td>
    <input    type="text"    
    name="notas[<?=    $aluno_id    ?>][nee]"    
    id="nee_<?=    $aluno_id    ?>"    
    value="<?=    $nee_salvo    >    0    ?    str_replace('.',    ',',    $nee_salvo)    :    ''    ?>"    
    class="nota-input    text-center    <?=    $nee_classe    ?>    nee-input"    
    maxlength="5"    placeholder="0,00"    
    data-aluno-id="<?=    $aluno_id    ?>"
    onkeypress="return    onlyNumbersAndComma(event)">
    </td>
    
    <td    id="mec_<?=    $aluno_id    ?>"    
    data-nota-valor="<?=    $mec    >    0    ?    number_format($mec,2,',','.'):'---'    ?>"    
    class="print-value    nota-calculada    <?=    $mec_classe    ?>">
    <?=    $mec    >    0    ?    number_format($mec,2,',','.'):'---'    ?>
    </td>
    <td    id="mf_<?=    $aluno_id    ?>"    
    data-nota-valor="<?=    $mf_display    ?>"    
    class="print-value    nota-final-calc    <?=    $mf_classe    ?>">
    <?=    $mf_display    ?>
    </td>
    <td    id="estado_<?=    $aluno_id    ?>"    
    data-nota-valor="<?=    $estado_final    ?>"    
    class="print-value    <?=    $estado_classe    ?>">
    <?=    $estado_final    ?>
    </td>
    <?php    endif;    ?>

    <td>
    <input    type="text"    
    name="notas[<?=    $aluno_id    ?>][observacao]"    
    value="<?=    htmlspecialchars($obs_salvo)    ?>"    
    class="text-center    w-full    nota-input"    
    maxlength="50">
    </td>
    </tr>
    <?php    endforeach;    ?>
    </tbody>
    </table>
    </div>

   <div class="no-print mt-6 flex justify-between flex-wrap gap-2">
    <button type="submit"
        class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded focus:outline-none focus:shadow-outline">
        Salvar Notas
    </button>
    <button type="button" onclick="window.print()"
        class="bg-gray-500 hover:bg-gray-700 text-white font-bold py-2 px-4 rounded">
        Imprimir Mini Pauta
    </button>
    <button type="button" onclick="window.location.href='caderneta_professor.php'"
        class="bg-yellow-500 hover:bg-yellow-600 text-white font-bold py-2 px-4 rounded">
        Voltar para Caderneta
    </button>
    <button type="button" onclick="window.location.href='painel_professor.php'"
        class="bg-green-500 hover:bg-green-600 text-white font-bold py-2 px-4 rounded">
        Ir para Painel
    </button>
</div>

    </form>
    <?php    endif;    ?>

</div>

<script>
    //    ⭐️    REVISÃO:    Função    de    validação    e    formatação    de    números
    function    formatNumber(value)    {
    //    Substitui    vírgula    por    ponto    para    cálculo    e    garante    que    seja    um    float
    let    cleanValue    =    String(value).replace(',',    '.');
    let    floatValue    =    parseFloat(cleanValue);
    
    //    Verifica    se    o    valor    é    válido    e    está    no    intervalo    [0,    20]
    if    (isNaN(floatValue)    ||    floatValue    <    0    ||    floatValue    >    20)    {
    return    0.0;    //    Retorna    0.0    se    for    inválido
    }
    return    floatValue;
    }

    //    ⭐️    REVISÃO:    Função    de    Recálculo    de    MT
    function    calcularMT(alunoId)    {
    const    macInput    =    document.querySelector(`input[name="mac_${alunoId}"]`);
    const    nppInput    =    document.querySelector(`input[name="notas[${alunoId}][npp]"]`);
    const    nptInput    =    document.querySelector(`input[name="notas[${alunoId}][npt]"]`);
    const    mtDisplay    =    document.querySelector(`#mt_display_${alunoId}`);

    //    1.    Obter    valores    limpos    e    garantir    0.0    se    inválido
    const    mac    =    formatNumber(macInput    ?    macInput.value    :    '0');
    const    npp    =    formatNumber(nppInput    ?    nppInput.value    :    '0');
    const    npt    =    formatNumber(nptInput    ?    nptInput.value    :    '0');
    
    //    2.    Calcular    MT:    (MAC    +    NPP    +    NPT)    /    3
    let    mt    =    0.0;
    if    (mac    >    0    ||    npp    >    0    ||    npt    >    0)    {
    mt    =    (mac    +    npp    +    npt)    /    3.0;
    }

    //    3.    Arredondar    para    duas    casas    decimais
    mt    =    Math.round(mt    *    100)    /    100;
    
    //    4.    Formatar    para    exibição    com    vírgula
    let    displayValue    =    mt    >    0    ?    mt.toFixed(2).replace('.',    ',')    :    '';
    
    //    5.    Atualizar    o    campo    de    exibição
    if    (mtDisplay)    {
    mtDisplay.value    =    displayValue;
    
    //    Atualizar    a    cor    de    fundo    (visual)
    const    mtClasse    =    mt    >=    10.0    ?    'nota-aprovada'    :    'nota-reprovada';
    mtDisplay.className    =    `nota-input    text-center    mt-display    ${mtClasse}`;
    
    //    Atualizar    o    atributo    para    impressão
    const    spanPrint    =    mtDisplay.closest('.print-value');
    if(spanPrint)    {
    spanPrint.setAttribute('data-nota-valor',    mt    >    0    ?    mt.toFixed(2).replace('.',    ',')    :    '---');
    spanPrint.className    =    `print-value    nota-calculada    ${mtClasse}`;
    }
    }
    }

    //    Função    de    validação    de    input    (permitir    apenas    números    e    vírgula)
    function    onlyNumbersAndComma(e)    {
    const    char    =    String.fromCharCode(e.keyCode);
    if    (/^[0-9,]+$/.test(char)    ||    e.keyCode    ===    8    ||    e.keyCode    ===    46)    {
    return    true;
    }
    e.preventDefault();
    return    false;
    }
    
    //    Adicionar    listeners    para    recálculo
    document.addEventListener('DOMContentLoaded',    function()    {
    const    inputs    =    document.querySelectorAll('.mac-input,    .nota-npp-input,    .nota-npt-input');
    inputs.forEach(input    =>    {
    input.addEventListener('input',    function()    {
    const    alunoId    =    this.getAttribute('data-aluno-id');
    //    Se    o    valor    estiver    vazio,    assume    0    para    cálculo,    mas    deixa    o    input    visualmente    vazio
    if    (this.value    ===    '')    {
    this.value    =    '';    
    }
    
    //    Limitar    o    valor    a    20    e    garantir    formato    float    no    cálculo
    let    floatValue    =    formatNumber(this.value);
    
    //    Recalcula    MT
    calcularMT(alunoId);
    });
    });
    
    //    Inicializar    os    cálculos    de    MT    na    carga    da    página
    document.querySelectorAll('input[name="aluno_id[]"]').forEach(input    =>    {
    calcularMT(input.value);
    });

    //    Adicionar    a    função    de    validação    aos    inputs    de    nota
    document.querySelectorAll('.nota-input').forEach(input    =>    {
    input.onkeypress    =    onlyNumbersAndComma;
    });
    });
</script>

</body>
</html>